<?php

namespace Bongo\User\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\User\Http\Requests\StoreUserRequest;
use Bongo\User\Http\Requests\UpdateUserRequest;
use Bongo\User\Models\User;
use Illuminate\Support\Facades\Hash;

class UserController extends AbstractController
{
    protected User $user;

    public function __construct(User $user)
    {
        $this->user = $user;
    }

    public function index()
    {
        return view('user::backend.index');
    }

    public function create()
    {
        $user = $this->user;

        return view('user::backend.create', compact('user'));
    }

    public function store(StoreUserRequest $request)
    {
        // Create the user
        $user = $this->user;
        $user->fill($request->except('password'));
        $user->password = Hash::make($request->get('password'));
        $user->save();

        return redirect()
            ->route('backend.user.show', $user->id)
            ->success(trans('user::backend.store_success'));
    }

    public function show(User $user)
    {
        if (! user()->isDeveloper() && $user->isDeveloper()) {
            abort(404);
        }

        return view('user::backend.show', compact('user'));
    }

    public function edit(User $user)
    {
        if (! user()->isDeveloper() && $user->isDeveloper()) {
            abort(404);
        }

        return view('user::backend.edit', compact('user'));
    }

    public function update(UpdateUserRequest $request, User $user)
    {
        // Update the user details
        $user->fill($request->except('password'));
        if ($request->has('password') && ! empty($request->get('password'))) {
            $user->password = Hash::make($request->get('password'));
        }
        $user->save();

        return redirect()
            ->route('backend.user.show', $user->id)
            ->success(trans('user::backend.update_success'));
    }

    public function destroy(User $user)
    {
        // Ensure the current user can't delete them self
        if (($user->id == user()->id) || (! user()->isDeveloper() && ! $user->isVisible())) {
            return back()->error(trans('user::backend.delete_failed'));
        }

        // finally delete the user
        if ($user->delete()) {
            $redirect = back();

            // If the user was deleted from their detail view,
            // we'll redirect back to the user list view.
            if (route('backend.user.show', $user) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.user.index');
            }

            return $redirect->success(trans('user::backend.delete_success'));
        }

        return back()->error(trans('user::backend.delete_failed'));
    }
}
