# Architecture Documentation - Bongo User Package

## Overview

The `bongo/user` package provides a complete user authentication, authorization, and management system for Laravel applications built on the Bongo framework. It implements role-based access control, user impersonation, and integrates with Laravel's authentication system while extending the Bongo framework's conventions.

## Directory Structure

```
src/
├── Commands/                              # Artisan commands
│   └── ResetPasswordCommand.php           # CLI: user:reset_password
│
├── Events/                                # Event handlers
│   └── UserEventHandler.php               # Cache management on user save
│
├── Factories/                             # Model factories
│   └── UserFactory.php                    # User factory for testing
│
├── Http/
│   ├── Controllers/
│   │   ├── Api/                           # API endpoints
│   │   │   └── UserController.php         # RESTful user API
│   │   │
│   │   ├── Auth/                          # Authentication flow
│   │   │   ├── AuthenticatedSessionController.php
│   │   │   ├── ConfirmablePasswordController.php
│   │   │   ├── EmailVerificationNotificationController.php
│   │   │   ├── EmailVerificationPromptController.php
│   │   │   ├── NewPasswordController.php
│   │   │   ├── PasswordController.php
│   │   │   ├── PasswordResetLinkController.php
│   │   │   ├── RegisteredUserController.php
│   │   │   └── VerifyEmailController.php
│   │   │
│   │   └── Backend/                       # Admin user management
│   │       ├── UserController.php         # CRUD operations
│   │       ├── UserDatatableController.php # Datatable data source
│   │       ├── UserImageController.php    # Avatar upload
│   │       └── UserImpersonationController.php # Impersonation
│   │
│   ├── Middleware/                        # Authorization middleware
│   │   ├── Developer.php                  # Developer-only access
│   │   ├── Employee.php                   # Employee+ access
│   │   └── Manager.php                    # Manager+ access
│   │
│   ├── Requests/                          # Form validation
│   │   ├── LoginRequest.php
│   │   ├── StoreUserRequest.php
│   │   └── UpdateUserRequest.php
│   │
│   ├── Resources/                         # API transformers
│   │   └── UserResource.php
│   │
│   └── ViewComposers/                     # View data injection
│       └── UserComposer.php               # Injects cached user list
│
├── Mailables/                             # Email classes
│   └── ResetPasswordMailable.php
│
├── Migrations/                            # Database migrations
│   ├── 2018_01_02_000001_create_users_table.php
│   └── 2018_01_02_000002_create_password_reset_tokens_table.php
│
├── Models/                                # Eloquent models
│   └── User.php                           # Main user model
│
├── Notifications/                         # Notification classes
│   └── ResetPasswordNotification.php      # Queued reset password email
│
├── Routes/                                # Route definitions
│   ├── api.php                            # API routes (auth:sanctum)
│   ├── backend.php                        # Admin routes (auth + manager)
│   └── web.php                            # Public auth routes
│
├── Traits/                                # Reusable traits
│   ├── HasAvatar.php                      # Avatar image support
│   ├── HasImpersonate.php                 # User impersonation
│   ├── HasName.php                        # Name helper methods
│   ├── HasPhone.php                       # Phone number helpers
│   └── HasType.php                        # User type scopes
│
├── Translations/                          # Language files
│   └── en/                                # English translations
│
├── Views/                                 # Blade templates
│   ├── auth/                              # Authentication views
│   ├── backend/                           # Admin user management
│   │   └── partials/
│   │       ├── dropdowns/
│   │       └── form/
│   └── mail/                              # Email templates
│
└── UserServiceProvider.php                # Package service provider
```

## Class Hierarchy Diagrams

### User Model Inheritance

```
Illuminate\Foundation\Auth\User (Authenticatable)
    │
    └── Bongo\User\Models\User
            │
            ├── Uses: HasApiTokens (Laravel\Sanctum)
            ├── Uses: Notifiable (Illuminate\Notifications)
            ├── Uses: SoftDeletes (Illuminate\Database\Eloquent)
            ├── Uses: HasUUID (Bongo\Framework\Traits)
            ├── Uses: HasStatus (Bongo\Framework\Traits)
            ├── Uses: HasImages (Bongo\Image\Traits)
            ├── Uses: HasAvatar (Bongo\User\Traits)
            ├── Uses: HasImpersonate (Bongo\User\Traits)
            ├── Uses: HasName (Bongo\User\Traits)
            ├── Uses: HasPhone (Bongo\User\Traits)
            └── Uses: HasType (Bongo\User\Traits)
```

### Controller Inheritance

```
Bongo\Framework\Http\Controllers\AbstractController
    │
    ├── Bongo\User\Http\Controllers\Backend\UserController
    ├── Bongo\User\Http\Controllers\Backend\UserDatatableController
    ├── Bongo\User\Http\Controllers\Backend\UserImageController
    ├── Bongo\User\Http\Controllers\Backend\UserImpersonationController
    │
    ├── Bongo\User\Http\Controllers\Auth\AuthenticatedSessionController
    ├── Bongo\User\Http\Controllers\Auth\RegisteredUserController
    ├── Bongo\User\Http\Controllers\Auth\PasswordResetLinkController
    ├── Bongo\User\Http\Controllers\Auth\NewPasswordController
    ├── Bongo\User\Http\Controllers\Auth\EmailVerificationPromptController
    ├── Bongo\User\Http\Controllers\Auth\VerifyEmailController
    ├── Bongo\User\Http\Controllers\Auth\PasswordController
    └── Bongo\User\Http\Controllers\Auth\ConfirmablePasswordController
```

### Service Provider Inheritance

```
Bongo\Framework\Providers\AbstractServiceProvider
    │
    └── Bongo\User\UserServiceProvider
            │
            ├── $module = 'user'
            ├── $middlewares = ['developer', 'manager', 'employee']
            ├── $commands = [ResetPasswordCommand]
            ├── $composers = [UserComposer]
            └── $subscribers = [UserEventHandler]
```

### Event Handler Inheritance

```
Bongo\Framework\Events\AbstractEventHandler
    │
    └── Bongo\User\Events\UserEventHandler
            │
            └── $model = User::class
```

## Authentication & Authorization Flow

### Login Flow

```
┌─────────────┐
│   Browser   │
└──────┬──────┘
       │ GET /login
       ▼
┌─────────────────────────────────────┐
│ AuthenticatedSessionController      │
│   ::create()                        │
│   → Returns login view              │
└──────┬──────────────────────────────┘
       │ POST /login (email, password)
       ▼
┌─────────────────────────────────────┐
│ LoginRequest                        │
│   → Validates credentials           │
│   → Authenticates user              │
└──────┬──────────────────────────────┘
       │
       ▼
┌─────────────────────────────────────┐
│ AuthenticatedSessionController      │
│   ::store()                         │
│   → Regenerates session             │
│   → Redirects to HOME               │
└─────────────────────────────────────┘
```

### Authorization Middleware Flow

```
                    ┌─────────────────┐
                    │ Incoming Request│
                    └────────┬────────┘
                             │
                ┌────────────▼───────────┐
                │  Is user logged in?    │
                └─┬──────────────────┬───┘
             NO   │                  │ YES
     ┌────────────▼──────┐           │
     │ Redirect to /login│           │
     └───────────────────┘           │
                          ┌──────────▼────────────┐
                          │ Check User Type       │
                          └─┬──────────┬─────────┬┘
                            │          │         │
              ┌─────────────▼───┐  ┌──▼───┐  ┌──▼────────┐
              │ developer       │  │manager│  │ employee  │
              │ middleware      │  │middle-│  │ middleware│
              │ isDeveloper()?  │  │ware   │  │           │
              └─────┬───────┬───┘  └┬─────┬┘  └┬─────────┬┘
                 NO │       │ YES   │     │     │         │
       ┌────────────▼┐      │       │     │     │         │
       │ abort(404)  │      │       │     │     │         │
       └─────────────┘      │       │     │     │         │
                            │       │     │     │         │
                     ┌──────▼───────▼─────▼─────▼─────────▼────┐
                     │        Continue to Controller           │
                     └──────────────────────────────────────────┘
```

### User Type Hierarchy

```
┌──────────────────────────────────────────────────────┐
│                    DEVELOPER                         │
│  • Full system access                                │
│  • Can manage all users including other developers   │
│  • Access: developer, manager, employee middleware   │
└───────────────────┬──────────────────────────────────┘
                    │
        ┌───────────▼──────────────────────────────────┐
        │                  MANAGER                      │
        │  • Can manage users (except developers)       │
        │  • Backend admin access                       │
        │  • Access: manager, employee middleware       │
        └───────────────────┬───────────────────────────┘
                            │
                ┌───────────▼──────────────────────────┐
                │              EMPLOYEE                 │
                │  • Standard backend access            │
                │  • Cannot manage users                │
                │  • Access: employee middleware        │
                └───────────────────────────────────────┘
```

## User Impersonation Architecture

### Impersonation Flow

```
┌────────────────────────────────────────────────────────┐
│ Admin User (Developer/Manager)                         │
└───────┬────────────────────────────────────────────────┘
        │ GET /admin/users/{user}/start-impersonating
        ▼
┌────────────────────────────────────────────────────────┐
│ UserImpersonationController::start()                   │
│   1. session()->put('impersonator', currentUser)       │
│   2. session()->put('impersonating', targetUser)       │
│   3. Auth::loginUsingId(targetUser->id)                │
└───────┬────────────────────────────────────────────────┘
        │
        ▼
┌────────────────────────────────────────────────────────┐
│ User now sees system as target user                    │
│ • All permissions reflect target user type             │
│ • Can access pages target user can access              │
│ • session('impersonating') contains target user        │
└───────┬────────────────────────────────────────────────┘
        │ GET /admin/users/stop-impersonating
        ▼
┌────────────────────────────────────────────────────────┐
│ UserImpersonationController::stop()                    │
│   1. Auth::loginUsingId(session('impersonator')->id)   │
│   2. session()->forget('impersonator')                 │
│   3. session()->forget('impersonating')                │
└───────┬────────────────────────────────────────────────┘
        │
        ▼
┌────────────────────────────────────────────────────────┐
│ Admin user restored to original session                │
└────────────────────────────────────────────────────────┘
```

### Session State During Impersonation

```
Before Impersonation:
session = {
    'user_id' => 1 (admin),
    // no impersonation keys
}

During Impersonation:
session = {
    'user_id' => 5 (target user),
    'impersonator' => User{id: 1} (admin),
    'impersonating' => User{id: 5} (target)
}

After Stop Impersonating:
session = {
    'user_id' => 1 (admin),
    // impersonation keys removed
}
```

## Routing Architecture

### Route Registration via AbstractServiceProvider

The `UserServiceProvider` extends `AbstractServiceProvider`, which automatically registers routes:

```
UserServiceProvider::boot()
    │
    └── AbstractServiceProvider::boot()
            │
            ├── Registers api.php
            │   ├── Prefix: /api/
            │   ├── Middleware: ['api', 'auth:sanctum']
            │   └── Name: api.user.*
            │
            ├── Registers backend.php
            │   ├── Prefix: /admin/
            │   ├── Middleware: ['web', 'auth', 'employee']
            │   └── Name: backend.user.*
            │
            └── Registers web.php
                ├── Prefix: /
                ├── Middleware: ['web']
                └── Named routes (login, password.reset, etc.)
```

### Route Groups and Middleware

**API Routes** (`src/Routes/api.php`):
```
/api/users                    [auth:sanctum]
    └── GET / → UserController@index
```

**Backend Routes** (`src/Routes/backend.php`):
```
/admin/users                  [auth, manager]
    ├── GET / → UserController@index
    ├── GET /create → UserController@create
    ├── POST /store → UserController@store
    ├── GET /datatable → UserDatatableController@index
    ├── GET /stop-impersonating → UserImpersonationController@stop
    └── /{user}/
        ├── GET / → UserController@show
        ├── GET /edit → UserController@edit
        ├── POST /update → UserController@update
        ├── DELETE /delete → UserController@destroy
        ├── POST /image → UserImageController@upload
        └── GET /start-impersonating → UserImpersonationController@start
```

**Web Routes** (`src/Routes/web.php`):
```
/                             [web]
    ├── /register             [guest]
    ├── /login                [guest]
    ├── /forgot-password      [guest]
    ├── /reset-password/{token} [guest]
    ├── /verify-email         [auth]
    ├── /confirm-password     [auth]
    └── /logout               [auth]
```

## Trait Reference Table

| Trait | Source Package | Purpose | Key Methods/Scopes |
|-------|----------------|---------|-------------------|
| `HasApiTokens` | Laravel Sanctum | API token authentication | `tokens()`, `createToken()`, `currentAccessToken()` |
| `Notifiable` | Laravel | Email/SMS notifications | `notify()`, `routeNotificationFor()` |
| `SoftDeletes` | Laravel | Soft delete support | `delete()`, `restore()`, `forceDelete()`, `trashed()` |
| `HasUUID` | bongo/framework | UUID generation | Automatically generates UUID on create |
| `HasStatus` | bongo/framework | Status management | `scopeActive()`, `scopeInactive()`, `scopePending()` |
| `HasImages` | bongo/image | Image relationships | `images()`, `getMainImage()` |
| `HasAvatar` | bongo/user | Avatar image support | `getAvatar()` |
| `HasImpersonate` | bongo/user | User impersonation | `startImpersonating()`, `stopImpersonating()`, `isImpersonating()` |
| `HasName` | bongo/user | Name helpers | `getFullName()`, name accessors |
| `HasPhone` | bongo/user | Phone helpers | Phone number formatting |
| `HasType` | bongo/user | User type scopes | `scopeDeveloper()`, `scopeManager()`, `scopeEmployee()`, `isDeveloper()`, `isManager()`, `isEmployee()` |

## Database Schema

### users Table

| Column | Type | Constraints | Description |
|--------|------|-------------|-------------|
| `id` | INT | PRIMARY KEY, AUTO INCREMENT | Primary identifier |
| `uuid` | VARCHAR(36) | INDEXED, NOT NULL | UUID identifier |
| `first_name` | VARCHAR(255) | NULLABLE | User's first name |
| `last_name` | VARCHAR(255) | NULLABLE | User's last name |
| `telephone` | VARCHAR(255) | NULLABLE | Landline number |
| `mobile` | VARCHAR(255) | NULLABLE | Mobile number |
| `email` | VARCHAR(255) | INDEXED, NOT NULL | Email address |
| `password` | VARCHAR(255) | NOT NULL | Hashed password |
| `remember_token` | VARCHAR(100) | NULLABLE | Remember me token |
| `email_verified_at` | TIMESTAMP | NULLABLE | Email verification timestamp |
| `type` | ENUM | NOT NULL, DEFAULT 'employee' | User type: developer, manager, employee |
| `status` | ENUM | NOT NULL, DEFAULT 'active' | User status: pending, active, inactive |
| `created_by` | INT | INDEXED, NULLABLE | User ID who created |
| `updated_by` | INT | INDEXED, NULLABLE | User ID who updated |
| `deleted_by` | INT | INDEXED, NULLABLE | User ID who deleted |
| `created_at` | TIMESTAMP | NULLABLE | Creation timestamp |
| `updated_at` | TIMESTAMP | NULLABLE | Last update timestamp |
| `deleted_at` | TIMESTAMP | NULLABLE | Soft delete timestamp |

### password_reset_tokens Table

| Column | Type | Constraints | Description |
|--------|------|-------------|-------------|
| `email` | VARCHAR(255) | PRIMARY KEY | User email |
| `token` | VARCHAR(255) | NOT NULL | Reset token |
| `created_at` | TIMESTAMP | NULLABLE | Token creation time |

## Extension Points

### Adding New User Types

1. **Update User Model Constants**:
```php
// src/Models/User.php
public const ADMIN = 'admin';
```

2. **Update Migration**:
```php
// src/Migrations/...create_users_table.php
$table->enum('type', [User::DEVELOPER, User::MANAGER, User::EMPLOYEE, User::ADMIN]);
```

3. **Add Trait Methods**:
```php
// src/Traits/HasType.php
public function scopeAdmin($query)
{
    return $query->where('type', self::ADMIN);
}

public function isAdmin(): bool
{
    return $this->type === self::ADMIN;
}
```

4. **Create Middleware**:
```php
// src/Http/Middleware/Admin.php
class Admin
{
    public function handle($request, Closure $next)
    {
        if (!user() || !user()->isAdmin()) {
            abort(404);
        }
        return $next($request);
    }
}
```

5. **Register Middleware**:
```php
// src/UserServiceProvider.php
protected array $middlewares = [
    'developer' => Developer::class,
    'manager' => Manager::class,
    'employee' => Employee::class,
    'admin' => Admin::class, // Add new middleware
];
```

### Adding New User Fields

1. **Create Migration**:
```php
Schema::table('users', function (Blueprint $table) {
    $table->string('company')->nullable()->after('mobile');
});
```

2. **Update Model**:
```php
// src/Models/User.php
protected $fillable = [
    'first_name', 'last_name', 'telephone', 'mobile', 'email', 'password', 'type',
    'company', // Add new field
];
```

3. **Update Form Requests**:
```php
// src/Http/Requests/StoreUserRequest.php
public function rules(): array
{
    return [
        // ... existing rules
        'company' => ['nullable', 'string', 'max:255'],
    ];
}
```

4. **Update Views**:
```blade
<!-- src/Views/backend/partials/form/_company.blade.php -->
<input type="text" name="company" value="{{ old('company', $user->company) }}">
```

5. **Update API Resource** (if needed):
```php
// src/Http/Resources/UserResource.php
public function toArray(Request $request): array
{
    return [
        // ... existing fields
        'company' => $this->company,
    ];
}
```

### Adding New Authentication Methods

1. **Create Controller**:
```php
// src/Http/Controllers/Auth/SocialAuthController.php
class SocialAuthController extends AbstractController
{
    public function redirectToProvider(string $provider)
    {
        return Socialite::driver($provider)->redirect();
    }

    public function handleProviderCallback(string $provider)
    {
        $socialUser = Socialite::driver($provider)->user();
        // ... authentication logic
    }
}
```

2. **Add Routes**:
```php
// src/Routes/web.php
Route::get('auth/{provider}', [SocialAuthController::class, 'redirectToProvider'])
    ->middleware('guest');
Route::get('auth/{provider}/callback', [SocialAuthController::class, 'handleProviderCallback'])
    ->middleware('guest');
```

3. **Update User Model** (add social provider fields):
```php
protected $fillable = [
    'first_name', 'last_name', 'email', 'password', 'type',
    'provider', 'provider_id', 'provider_token',
];
```

### Extending with Events

The package uses `UserEventHandler` to react to model events. Extend this pattern:

1. **Create New Handler**:
```php
// src/Events/UserActivityHandler.php
class UserActivityHandler extends AbstractEventHandler
{
    protected string $model = User::class;

    public function onCreate($model): void
    {
        // Log user creation
        activity('user')->log('User created: ' . $model->email);
    }

    public function onUpdate($model): void
    {
        // Log user updates
        activity('user')->log('User updated: ' . $model->email);
    }
}
```

2. **Register Handler**:
```php
// src/UserServiceProvider.php
protected array $subscribers = [
    UserEventHandler::class,
    UserActivityHandler::class, // Add new handler
];
```

## How to Add New Features

### Adding a User Profile Page

1. **Create Controller Method**:
```php
// src/Http/Controllers/Backend/UserProfileController.php
class UserProfileController extends AbstractController
{
    public function show()
    {
        $user = auth()->user();
        return view('user::backend.profile', compact('user'));
    }

    public function update(UpdateProfileRequest $request)
    {
        $user = auth()->user();
        $user->fill($request->validated());
        $user->save();

        return redirect()->route('backend.user.profile')
            ->success('Profile updated successfully');
    }
}
```

2. **Add Routes**:
```php
// src/Routes/backend.php
Route::get('profile', [UserProfileController::class, 'show'])
    ->name('profile');
Route::post('profile', [UserProfileController::class, 'update'])
    ->name('profile.update');
```

3. **Create Views**:
```blade
<!-- src/Views/backend/profile.blade.php -->
@extends('framework::backend.layouts.master')

@section('content')
    <form action="{{ route('backend.user.profile.update') }}" method="POST">
        @csrf
        <!-- profile form fields -->
    </form>
@endsection
```

### Adding Two-Factor Authentication

1. **Install Package**:
```bash
composer require pragmarx/google2fa-laravel
```

2. **Add Migration**:
```php
Schema::table('users', function (Blueprint $table) {
    $table->string('two_factor_secret')->nullable();
    $table->timestamp('two_factor_enabled_at')->nullable();
});
```

3. **Create Trait**:
```php
// src/Traits/HasTwoFactorAuth.php
trait HasTwoFactorAuth
{
    public function hasTwoFactorEnabled(): bool
    {
        return !is_null($this->two_factor_enabled_at);
    }

    public function enableTwoFactor(string $secret): void
    {
        $this->two_factor_secret = encrypt($secret);
        $this->two_factor_enabled_at = now();
        $this->save();
    }

    public function disableTwoFactor(): void
    {
        $this->two_factor_secret = null;
        $this->two_factor_enabled_at = null;
        $this->save();
    }
}
```

4. **Use Trait in User Model**:
```php
class User extends Authenticatable
{
    use HasTwoFactorAuth;
    // ... other traits
}
```

5. **Create Controllers and Routes** for 2FA setup/verification.

## Cache Strategy

The package implements caching in two places:

1. **UserComposer** (src/Http/ViewComposers/UserComposer.php:16):
   - Caches user list for dropdowns
   - Cache key: `'users'`
   - Cache duration: `config('settings.cache_default')`
   - Cleared by: `UserEventHandler::onSave()`

2. **UserEventHandler** (src/Events/UserEventHandler.php:17):
   - Clears `'packages'` cache on user save
   - Ensures user changes propagate to dependent caches

To add new caching:

```php
// In a view composer or controller
$data = Cache::remember('cache_key', $duration, function () {
    return ExpensiveQuery::all();
});

// In UserEventHandler, clear it on user save
public function onSave($model): void
{
    parent::onSave($model);
    Cache::forget('cache_key');
}
```

## Testing Strategy

The package includes a `TestCase` (tests/TestCase.php) that extends Orchestra Testbench.

### Testing User Creation

```php
public function test_can_create_user()
{
    $userData = [
        'first_name' => 'John',
        'last_name' => 'Doe',
        'email' => 'john@example.com',
        'password' => 'password123',
        'type' => User::EMPLOYEE,
    ];

    $user = User::create([
        ...$userData,
        'password' => Hash::make($userData['password']),
    ]);

    $this->assertDatabaseHas('users', [
        'email' => 'john@example.com',
        'type' => User::EMPLOYEE,
    ]);
}
```

### Testing Authentication

```php
public function test_user_can_login()
{
    $user = User::factory()->create([
        'email' => 'test@example.com',
        'password' => Hash::make('password'),
    ]);

    $response = $this->post('/login', [
        'email' => 'test@example.com',
        'password' => 'password',
    ]);

    $response->assertRedirect('/');
    $this->assertAuthenticatedAs($user);
}
```

### Testing Authorization

```php
public function test_employee_cannot_access_user_management()
{
    $employee = User::factory()->create(['type' => User::EMPLOYEE]);

    $this->actingAs($employee)
        ->get('/admin/users')
        ->assertStatus(404);
}

public function test_manager_can_access_user_management()
{
    $manager = User::factory()->create(['type' => User::MANAGER]);

    $this->actingAs($manager)
        ->get('/admin/users')
        ->assertStatus(200);
}
```

### Testing Impersonation

```php
public function test_admin_can_impersonate_user()
{
    $admin = User::factory()->create(['type' => User::DEVELOPER]);
    $user = User::factory()->create(['type' => User::EMPLOYEE]);

    $this->actingAs($admin)
        ->get("/admin/users/{$user->id}/start-impersonating");

    $this->assertEquals($user->id, session('impersonating')->id);
    $this->assertEquals($admin->id, session('impersonator')->id);
    $this->assertTrue($admin->isImpersonating());
}
```

## Performance Considerations

1. **Soft Deletes**: Uses soft deletes to preserve data integrity. Query visible users with `User::visible()`
2. **Eager Loading**: When displaying user lists, eager load relationships to avoid N+1 queries
3. **Caching**: User list is cached for dropdowns. Cache invalidated on user save
4. **Indexing**: Email and UUID fields are indexed for faster lookups
5. **Password Hashing**: Uses Laravel's `Hash::make()` which uses bcrypt (secure but slow - hash once, check carefully)

## Security Considerations

1. **Developer Protection**: Non-developer users cannot view, edit, or delete developer accounts (404 error)
2. **Self-Protection**: Users cannot delete themselves
3. **Password Security**: Always hash passwords with `Hash::make()`
4. **Session Regeneration**: Sessions regenerated on login to prevent session fixation
5. **CSRF Protection**: All forms use `@csrf` token
6. **Route Middleware**: All backend routes require authentication and appropriate user type
7. **Email Verification**: Supports email verification flow
8. **Password Reset**: Uses secure token-based password reset
9. **Impersonation Safety**: Stop-impersonating route allowed even for restricted users

## Integration with Bongo Framework

This package tightly integrates with `bongo/framework`:

1. **Extends AbstractServiceProvider**: Inherits automatic bootstrapping
2. **Extends AbstractController**: Controllers inherit base functionality
3. **Extends AbstractEventHandler**: Event handler inherits model event listening
4. **Uses Framework Traits**: `HasUUID`, `HasStatus` from framework
5. **Uses Framework Patterns**: Follows framework conventions for routing, views, translations

## Dependencies Graph

```
bongo/user
    ├── bongo/framework (required)
    │   ├── Providers: AbstractServiceProvider
    │   ├── Controllers: AbstractController
    │   ├── Events: AbstractEventHandler
    │   └── Traits: HasUUID, HasStatus
    │
    ├── bongo/image (used via HasImages trait)
    │   └── Traits: HasImages
    │
    ├── laravel/framework ^10.0 (via bongo/framework)
    │   ├── Authentication
    │   ├── Eloquent ORM
    │   └── Blade Templates
    │
    └── laravel/sanctum (for API authentication)
        └── HasApiTokens trait
```

This architecture provides a solid foundation for user management while remaining extensible and maintainable.
